// ...existing code...
import 'dotenv/config';
import express from 'express';
import session from 'express-session';
import helmet from 'helmet';
import path from 'path';
import multer from 'multer';
import mammoth from 'mammoth';
import { createRequire } from 'module';
const require = createRequire(import.meta.url);

// Robust PDF extraction: probar dinámicamente pdf-parse (ESM/CJS) y usar pdfjs-dist como fallback.
async function runPdfParse(buffer) {
  // 1) Intentar import dinámico (ESM)
  try {
    const mod = await import('pdf-parse').catch(() => null);
    if (mod) {
      const fn = mod.default || mod;
      if (typeof fn === 'function') {
        return await fn(buffer);
      }
      if (fn && typeof fn.parse === 'function') {
        return await fn.parse(buffer);
      }
    }
  } catch (err) {
    // continuar a siguiente estrategia
  }

  // 2) Intentar require (CJS) vía createRequire
  try {
    const _pdfParse = require('pdf-parse');
    const pdfParseCandidate = (typeof _pdfParse === 'function') ? _pdfParse
      : (_pdfParse && typeof _pdfParse.default === 'function') ? _pdfParse.default
      : (_pdfParse && typeof _pdfParse.parse === 'function') ? _pdfParse.parse
      : null;
    if (typeof pdfParseCandidate === 'function') {
      return await pdfParseCandidate(buffer);
    }
  } catch (err) {
    // seguir a fallback
  }

  // 3) Fallback: pdfjs-dist (legacy build para Node)
  try {
    const pdfjs = require('pdfjs-dist/legacy/build/pdf.js');
    const uint8 = new Uint8Array(buffer);
    const loadingTask = pdfjs.getDocument({ data: uint8 });
    const doc = await loadingTask.promise;
    let fullText = '';
    for (let i = 1; i <= doc.numPages; i++) {
      const page = await doc.getPage(i);
      const content = await page.getTextContent();
      const strs = content.items.map(it => (it.str || ''));
      fullText += strs.join(' ') + '\n\n';
    }
    return { text: fullText };
  } catch (err) {
    // nada más que intentar
  }

  throw new Error('No fue posible extraer texto del PDF. Instala "pdf-parse" (npm i pdf-parse) o "pdfjs-dist" (npm i pdfjs-dist).');
}

import { execSync } from 'child_process';
import bcrypt from 'bcrypt';
import SQLite from 'better-sqlite3';
import { fileURLToPath } from 'url';
import { OpenAI } from 'openai';
import fs from 'fs';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

const app = express();
app.set('layout', false);

// Seguridad y básicos
app.use(helmet({
  contentSecurityPolicy: false
}));
app.use(express.urlencoded({ extended: true }));
app.use(express.json());
app.use('/public', express.static(path.join(__dirname, 'public')));
app.use('/assets', express.static(path.join(__dirname, 'assets')));
// Sessions
app.use(session({
  secret: process.env.SESSION_SECRET || 'devsecret',
  resave: false,
  saveUninitialized: false,
  cookie: { sameSite: 'lax' }
}));

// Vistas
app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));

// DB (SQLite, síncrono, simple)
const db = new SQLite(path.join(__dirname, 'app.db'));
db.pragma('journal_mode = WAL');

// Tablas
db.exec(`
CREATE TABLE IF NOT EXISTS users (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  email TEXT UNIQUE NOT NULL,
  name TEXT NOT NULL,
  password_hash TEXT NOT NULL,
  role TEXT NOT NULL DEFAULT 'user',
  created_at TEXT NOT NULL DEFAULT (datetime('now'))
);

CREATE TABLE IF NOT EXISTS chats (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  user_id INTEGER NOT NULL UNIQUE,
  messages_json TEXT NOT NULL DEFAULT '[]',
  last_updated TEXT NOT NULL DEFAULT (datetime('now')),
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

CREATE TABLE IF NOT EXISTS uploads (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  user_id INTEGER NOT NULL,
  filename TEXT NOT NULL,
  original_name TEXT NOT NULL,
  extracted_text TEXT,
  created_at TEXT NOT NULL DEFAULT (datetime('now')),
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

CREATE TABLE IF NOT EXISTS knowledge (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  filename TEXT NOT NULL,
  original_name TEXT NOT NULL,
  content TEXT NOT NULL,
  embedding TEXT,
  created_at TEXT NOT NULL DEFAULT (datetime('now'))
);
`);

// Crear admin por defecto (si no existe)
if (process.env.ADMIN_EMAIL && process.env.ADMIN_PASSWORD) {
  const getAdmin = db.prepare('SELECT * FROM users WHERE email = ?').get(process.env.ADMIN_EMAIL);
  if (!getAdmin) {
    const hash = bcrypt.hashSync(process.env.ADMIN_PASSWORD, 10);
    db.prepare('INSERT INTO users (email, name, password_hash, role) VALUES (?, ?, ?, ?)').run(
      process.env.ADMIN_EMAIL, 'Administrador', hash, 'admin'
    );
    const adminRow = db.prepare('SELECT id FROM users WHERE email = ?').get(process.env.ADMIN_EMAIL);
    db.prepare('INSERT OR IGNORE INTO chats (user_id, messages_json) VALUES (?, ?)').run(adminRow.id, '[]');
  }
}

// Multer para subidas
const uploadDir = path.join(__dirname, 'uploads');
// asegurar carpeta uploads
if (!fs.existsSync(uploadDir)) fs.mkdirSync(uploadDir, { recursive: true });

const storage = multer.diskStorage({
  destination: (req, file, cb) => cb(null, uploadDir),
  filename: (req, file, cb) => {
    const unique = Date.now() + '-' + Math.round(Math.random() * 1e9);
    cb(null, unique + path.extname(file.originalname));
  }
});
const upload = multer({
  storage,
  // permitir cualquier tipo de archivo — el servidor intentará extraer texto para formatos conocidos
  fileFilter: (req, file, cb) => cb(null, true),
  limits: { fileSize: 50 * 1024 * 1024 } // 50MB
});

// OpenAI client
const openai = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });

// Helpers
function requireAuth(req, res, next) {
  if (!req.session.user) {
    req.session.returnTo = req.originalUrl;
    return res.redirect('/login');
  }
  next();
}

function requireAdmin(req, res, next) {
  if (!req.session.user || req.session.user.role !== 'admin') return res.status(403).send('No autorizado');
  next();
}

function getUserByEmail(email) {
  return db.prepare('SELECT * FROM users WHERE email = ?').get(email);
}

function getUserById(id) {
  return db.prepare('SELECT * FROM users WHERE id = ?').get(id);
}

function ensureUserChat(userId) {
  const row = db.prepare('SELECT * FROM chats WHERE user_id = ?').get(userId);
  if (!row) {
    db.prepare('INSERT INTO chats (user_id, messages_json) VALUES (?, ?)').run(userId, '[]');
    return db.prepare('SELECT * FROM chats WHERE user_id = ?').get(userId);
  }
  return row;
}

function saveChatMessages(userId, messages) {
  db.prepare('UPDATE chats SET messages_json = ?, last_updated = datetime(\'now\') WHERE user_id = ?')
    .run(JSON.stringify(messages), userId);
}

// Extract text helper supporting docx, doc, pdf, txt and fallback attempts
async function extractTextFromFile(fullPath, originalName) {
  const ext = path.extname(originalName || fullPath || '').toLowerCase();

  try {
    if (ext === '.docx') {
      const result = await mammoth.extractRawText({ path: fullPath });
      return (result.value || '').trim();
    }

    if (ext === '.txt' || ext === '.md' || ext === '.csv') {
      return fs.readFileSync(fullPath, 'utf8');
    }

    if (ext === '.pdf') {
      const data = fs.readFileSync(fullPath);
      const parsed = await runPdfParse(data);
      return (parsed?.text || '').trim();
    }

    if (ext === '.doc') {
      const outDir = path.dirname(fullPath);
      try {
        execSync(`soffice --headless --convert-to docx --outdir "${outDir}" "${fullPath}"`, { stdio: 'ignore' });
        const converted = fullPath.replace(/\.doc$/i, '.docx');
        const result = await mammoth.extractRawText({ path: converted });
        try { fs.unlinkSync(converted); } catch (_) {}
        return (result.value || '').trim();
      } catch (err) {
        throw new Error('Error convirtiendo .doc (requiere libreoffice / "soffice"): ' + err.message);
      }
    }

    // Intento genérico: si el archivo no es binario grande, leer como utf8
    try {
      const maybeText = fs.readFileSync(fullPath, 'utf8');
      // si hay caracteres de control binarios en porcentaje alto, rechazamos
      const binCheck = /[\x00-\x08\x0E-\x1F\x7F]/;
      if (binCheck.test(maybeText)) {
        return '';
      }
      return maybeText.trim();
    } catch {
      return '';
    }
  } catch (err) {
    // propaga para manejo upstream
    throw err;
  }
}

async function evaluateThesisWithAI(thesisText) {
  const system = `Eres un evaluador académico de trabajos de grado. 
Devuelve un JSON con: 
- "score": número entero 0-100, 
- "reasons": lista breve de razones (3-6), 
- "recommendations": lista de recomendaciones accionables (3-6). 
Evalúa claridad, estructura, metodología, rigor, redacción, y originalidad con base SOLO en el texto proporcionado.`;
  const user = `Texto de la tesis (recortado si es muy largo):
"""${(thesisText || '').slice(0, 12000)}"""`;

  // Nota: la llamada aquí usa la API original del SDK usada en este proyecto.
  const resp = await openai.chat.completions.create({
    model: 'gpt-4o-mini',
    temperature: 0.2,
    messages: [
      { role: 'system', content: system },
      { role: 'user', content: user }
    ],
    response_format: { type: 'json_object' }
  });

  let parsed = { score: 0, reasons: [], recommendations: [] };
  try {
    parsed = JSON.parse(resp.choices[0].message.content);
  } catch { /* fallback */ }
  return parsed;
}

async function chatAskAI(history, userMessage) {
  const messages = [
    { role: 'system', content: 'Eres un asistente académico que ayuda a mejorar tesis. Sé claro, conciso y proactivo.' },
    ...history.map(m => ({ role: m.role, content: m.content })),
    { role: 'user', content: userMessage }
  ];
  const resp = await openai.chat.completions.create({
    model: 'gpt-4o-mini',
    temperature: 0.4,
    messages
  });
  return resp.choices[0].message.content;
}

// Rutas públicas
app.get('/', (req, res) => {
  if (req.session.user) return res.redirect('/welcome');
  res.render('welcome', { user: null });
});

app.get('/login', (req, res) => {
  res.render('login', { error: null });
});

app.post('/login', (req, res) => {
  const { email, password } = req.body;
  const user = getUserByEmail(email);
  if (!user) return res.render('login', { error: 'Credenciales inválidas' });
  if (!bcrypt.compareSync(password, user.password_hash)) {
    return res.render('login', { error: 'Credenciales inválidas' });
  }
  req.session.user = { id: user.id, email: user.email, name: user.name, role: user.role };
  ensureUserChat(user.id);
  const dest = req.session.returnTo || '/chat';
  delete req.session.returnTo;
  res.redirect(dest);
});

app.get('/register', (req, res) => {
  res.render('register', { error: null });
});

app.post('/register', (req, res) => {
  const { email, name, password } = req.body;
  if (!email || !name || !password) return res.render('register', { error: 'Completa todos los campos' });
  try {
    const hash = bcrypt.hashSync(password, 10);
    db.prepare('INSERT INTO users (email, name, password_hash) VALUES (?, ?, ?)').run(email, name, hash);
    const user = getUserByEmail(email);
    ensureUserChat(user.id);
    req.session.user = { id: user.id, email: user.email, name: user.name, role: user.role };
    const dest = req.session.returnTo || '/chat';
    delete req.session.returnTo;
    res.redirect(dest);
  } catch (e) {
    res.render('register', { error: 'El correo ya está registrado' });
  }
});

app.get('/logout', (req, res) => {
  req.session.destroy(() => res.redirect('/login'));
});

app.get('/welcome', (req, res) => {
  res.render('welcome', { user: req.session.user || null });
});

app.get('/chat', requireAuth, (req, res) => {
  const chat = ensureUserChat(req.session.user.id);
  const messages = JSON.parse(chat.messages_json);
  const uploads = db.prepare('SELECT * FROM uploads WHERE user_id = ? ORDER BY created_at DESC').all(req.session.user.id);
  res.render('chat', { user: req.session.user, messages, uploads, error: null });
});

app.post('/chat/clear', requireAuth, (req, res) => {
  saveChatMessages(req.session.user.id, []);
  res.redirect('/chat');
});

app.get('/uploads/:filename', requireAuth, (req, res) => {
  const filePath = path.join(uploadDir, req.params.filename);
  res.download(filePath);
});

app.post('/upload-doc', requireAuth, upload.single('doc'), async (req, res) => {
  try {
    const { file } = req;
    if (!file) return res.status(400).json({ ok: false, error: 'Archivo requerido' });

    const fullPath = path.join(uploadDir, file.filename);
    const text = await extractTextFromFile(fullPath, file.originalname);

    db.prepare('INSERT INTO uploads (user_id, filename, original_name, extracted_text) VALUES (?, ?, ?, ?)').run(
      req.session.user.id, file.filename, file.originalname, text
    );

    const evaluation = await evaluateThesisWithAI(text || '');

    const chat = ensureUserChat(req.session.user.id);
    const messages = JSON.parse(chat.messages_json);
    messages.push({
      role: 'system',
      content: `Evaluación automática:\n\nScore: ${evaluation.score}%\n\nRazones:\n- ${evaluation.reasons.join('\n- ')}\n\nRecomendaciones:\n- ${evaluation.recommendations.join('\n- ')}`,
      timestamp: Date.now()
    });
    saveChatMessages(req.session.user.id, messages);

    res.redirect('/chat');
  } catch (e) {
    console.error('Upload error:', e);
    if (req.session.user) {
      const chat = ensureUserChat(req.session.user.id);
      const messages = chat ? JSON.parse(chat.messages_json) : [];
      return res.render('chat', { user: req.session.user, messages, uploads: [], error: e.message });
    }
    res.status(500).send(e.message);
  }
});

app.post('/api/message', requireAuth, async (req, res) => {
  try {
    const { message } = req.body;
    if (!message || !message.trim()) return res.status(400).json({ ok: false, error: 'Mensaje vacío' });

    const chat = ensureUserChat(req.session.user.id);
    const history = JSON.parse(chat.messages_json);

    history.push({ role: 'user', content: message.trim(), timestamp: Date.now() });

    const aiReply = await chatAskAI(history, message.trim());

    history.push({ role: 'assistant', content: aiReply, timestamp: Date.now() });
    saveChatMessages(req.session.user.id, history);

    res.json({ ok: true, reply: aiReply });
  } catch (e) {
    res.status(500).json({ ok: false, error: e.message });
  }
});

app.get('/profile', requireAuth, (req, res) => {
  const user = getUserById(req.session.user.id);
  res.render('profile', { user, success: null, error: null });
});

app.post('/profile', requireAuth, (req, res) => {
  const { name, current_password, new_password } = req.body;
  const user = getUserById(req.session.user.id);

  if (name && name.trim()) {
    db.prepare('UPDATE users SET name = ? WHERE id = ?').run(name.trim(), user.id);
    req.session.user.name = name.trim();
  }

  if (new_password && new_password.length >= 6) {
    if (!current_password || !bcrypt.compareSync(current_password, user.password_hash)) {
      const fresh = getUserById(user.id);
      return res.render('profile', { user: fresh, success: null, error: 'Contraseña actual incorrecta' });
    }
    const hash = bcrypt.hashSync(new_password, 10);
    db.prepare('UPDATE users SET password_hash = ? WHERE id = ?').run(hash, user.id);
  }

  const fresh = getUserById(user.id);
  res.render('profile', { user: fresh, success: 'Perfil actualizado', error: null });
});

app.get('/admin/users', requireAuth, requireAdmin, (req, res) => {
  const users = db.prepare('SELECT id, email, name, role, created_at FROM users ORDER BY created_at DESC').all();
  const knowledge = db.prepare('SELECT id, original_name, filename, created_at FROM knowledge ORDER BY created_at DESC').all();
  res.render('admin_users', { me: req.session.user, users, knowledge, error: null, success: null });
});

app.post('/admin/users', requireAuth, requireAdmin, (req, res) => {
  const { action, user_id, role, temp_password } = req.body;
  try {
    if (action === 'make_admin') {
      db.prepare('UPDATE users SET role = ? WHERE id = ?').run('admin', user_id);
    } else if (action === 'make_user') {
      db.prepare('UPDATE users SET role = ? WHERE id = ?').run('user', user_id);
    } else if (action === 'reset_password') {
      if (!temp_password || temp_password.length < 6) throw new Error('Password temporal mínimo 6 chars');
      const hash = bcrypt.hashSync(temp_password, 10);
      db.prepare('UPDATE users SET password_hash = ? WHERE id = ?').run(hash, user_id);
    } else if (action === 'delete_user') {
      db.prepare('DELETE FROM users WHERE id = ?').run(user_id);
    }
    const users = db.prepare('SELECT id, email, name, role, created_at FROM users ORDER BY created_at DESC').all();
    const knowledge = db.prepare('SELECT id, original_name, filename, created_at FROM knowledge ORDER BY created_at DESC').all();
    res.render('admin_users', { me: req.session.user, users, knowledge, error: null, success: 'Acción ejecutada' });
  } catch (e) {
    const users = db.prepare('SELECT id, email, name, role, created_at FROM users ORDER BY created_at DESC').all();
    const knowledge = db.prepare('SELECT id, original_name, filename, created_at FROM knowledge ORDER BY created_at DESC').all();
    res.render('admin_users', { me: req.session.user, users, knowledge, error: e.message, success: null });
  }
});

app.post('/admin/knowledge/upload', requireAuth, requireAdmin, upload.single('doc'), async (req, res) => {
  try {
    const file = req.file;
    if (!file) {
      const users = db.prepare('SELECT id, email, name, role, created_at FROM users ORDER BY created_at DESC').all();
      const knowledge = db.prepare('SELECT id, original_name, filename, created_at FROM knowledge ORDER BY created_at DESC').all();
      return res.render('admin_users', { me: req.session.user, users, knowledge, error: 'Archivo requerido', success: null });
    }

    const fullPath = path.join(uploadDir, file.filename);
    const text = await extractTextFromFile(fullPath, file.originalname);

    const info = db.prepare('INSERT INTO knowledge (filename, original_name, content) VALUES (?, ?, ?)').run(
      file.filename, file.originalname, text || ''
    );

    const toEmbed = (text || '').slice(0, 24000);
    try {
      const embResp = await openai.embeddings.create({
        model: "text-embedding-3-small",
        input: toEmbed
      });
      const embedding = embResp.data?.[0]?.embedding || [];
      db.prepare('UPDATE knowledge SET embedding = ? WHERE id = ?').run(JSON.stringify(embedding), info.lastInsertRowid);
    } catch (errEmbedding) {
      console.error('Embedding error', errEmbedding);
    }

    res.redirect('/admin/users');
  } catch (err) {
    console.error('Knowledge upload error', err);
    const users = db.prepare('SELECT id, email, name, role, created_at FROM users ORDER BY created_at DESC').all();
    const knowledge = db.prepare('SELECT id, original_name, filename, created_at FROM knowledge ORDER BY created_at DESC').all();
    res.render('admin_users', { me: req.session.user, users, knowledge, error: err.message, success: null });
  }
});

app.post('/admin/knowledge/delete', requireAuth, requireAdmin, (req, res) => {
  const { id } = req.body;
  const row = db.prepare('SELECT filename FROM knowledge WHERE id = ?').get(id);
  if (row) {
    try { fs.unlinkSync(path.join(uploadDir, row.filename)); } catch (e) { /* ignore */ }
  }
  db.prepare('DELETE FROM knowledge WHERE id = ?').run(id);
  res.redirect('/admin/users');
});

app.post('/admin/clear-data', requireAuth, requireAdmin, (req, res) => {
  try {
    if (fs.existsSync(uploadDir)) {
      fs.readdirSync(uploadDir).forEach(file => {
        const p = path.join(uploadDir, file);
        try { fs.unlinkSync(p); } catch (err) { /* ignorar */ }
      });
    }

    db.exec(`
      DELETE FROM chats;
      DELETE FROM uploads;
      DELETE FROM knowledge;
      DELETE FROM users WHERE role != 'admin';
      VACUUM;
    `);

    res.redirect('/admin/users');
  } catch (err) {
    console.error('Error clearing data:', err);
    const users = db.prepare('SELECT id, email, name, role, created_at FROM users ORDER BY created_at DESC').all();
    const knowledge = db.prepare('SELECT id, original_name, filename, created_at FROM knowledge ORDER BY created_at DESC').all();
    res.render('admin_users', { me: req.session.user, users, knowledge, error: 'Error limpiando datos: ' + err.message, success: null });
  }
});

app.use((req, res) => res.status(404).send('No encontrado'));

const PORT = process.env.PORT || 3000;
app.listen(PORT, () => {
  console.log(`🚀 App en http://localhost:${PORT}`);
});
// ...existing code...